#!/usr/bin/env python3
# -*- coding: utf-8 -*-

'''
Programmname: Video-Umwandler
Dateiname: video-umwandler.py
Autor: Joachim Jakob
Lizenz: GPLv3

Zweck:
  Lokal auf einem PC oder Notebook mit einem Desktop-Betriebssystem 
  liegen bereits vorhandene Video-Dateien in verschiedenen 
  Video-Dateiformaten vor.
  Diese sollen z.B. online über eine Lernplattform oder einen 
  Cloudspeicher verbreitet werden.
  Dafür sollte die Dateigröße möglichst reduziert werden.
  Um ein Abspielen direkt im Browser mit HTML5 Video Elementen 
  zu erlauben, muss als kompatibles Format MP4 als Container mit
  AAC als Audio-Codec und H264 als Video-Codec eingesetzt werden.
  https://support.google.com/youtube/answer/6375112
  https://support.google.com/youtube/answer/1722171
  
  Dieses Programm erlaubt die Umwandlung, Skalierung, Komprimierung und
  Recodierung (fast) beliebiger Quell-Videodateiformate nach MP4.

Software-Voraussetzungen:
- Python 3
  https://www.python.org/downloads/
- ffmpeg-python: Python bindings for FFmpeg
  https://github.com/kkroening/ffmpeg-python
- FFmpeg
  https://www.ffmpeg.org/

Installation unter Ubuntu-Linux 20.04:
$ sudo apt install python3 ffmpeg
$ sudo pip3 install ffmpeg-python

Installation unter Windows (nur 64bit):
1) Herunterladen der aktuellsten 3er Version von 
   https://www.python.org/downloads/
2) im Installer auswählen: ☒ Add Python 3.7 to PATH
3) über die Windowssuche "cmd" eingeben und in der "schwarzen Box" dann python eingeben.
4) hinter dem Eingabeprompt > dann eingeben: 
   pip install ffmpeg-python
   und die Eingabetaste drücken
5) Herunterladen der aktuellsten Version von 
   https://www.ffmpeg.org/download.html#build-windows
6) Entpacken und das Download-Verzeichnis dem Pfad hinzufügen, 
   damit ffmpeg von dem Python-Wrapper gefunden werden kann

Entsprechender Kommandozeilenbefehl (Bash) unter Linux für den eigentlichen Umwandlungsschritt:
$ ffmpeg -i "${infile}" -movflags faststart -c:v libx264 -c:a aac -b:a 128k -s:v ${zielbreite}x${zielhoehe} "${outfile}.mp4"
'''

import importlib
import os
import sys
import pathlib
import glob
import subprocess
#import thread
from tkinter import *
from tkinter import messagebox
from tkinter.filedialog import askopenfilename
from tkinter.filedialog import askdirectory


try:
    import ffmpeg
except ModuleNotFoundError as err:
    fehler_ffmpegnichtgefunden = messagebox.showerror('⚠ FFMPEG nicht gefunden!', 'Es muss ffmpeg-python installiert werden.\nWeitere Informationen unter: https://github.com/kkroening/ffmpeg-python\nunter Ubuntu z.B. mit:\nsudo pip3 install ffmpeg-python')
    print(err)
    sys.exit(0)
    pass



class Videoumwandler:
    def __init__(self):
        ## Fenster-Einstellungen
        self.fenster = Tk()
        self.fenster.geometry('480x420')
        self.fenster.title('Video umwandeln')
        
        ## Converter-Einstellungen
        self.default_aufloesung='360p'
        self.aufloesung=StringVar()
        self.aufloesung.set(self.default_aufloesung)
        self.aufloesungen={
            '1080p':[1920, 1080],
            '720p':[1280, 720],
            '480p':[854, 480],
            '360p':[640, 360],
            '240p':[426, 240]
        }
        self.quelldateitypen=(
            ('alle Dateien','*.*'),
            ('MP4','*.mp4'),
            ('WEBM','*.webm'),
            ('OGG','*.ogg'),
            ('OGV','*.ogv'),
            ('MPG','*.mpg'),
            ('MPEG','*.mpeg'),
            ('AVI','*.avi'),
            ('WMV','*.wmv'),
            ('QT','*.qt'),
            ('MOV','*.mov'),
            ('MKV','*.mkv'),
            ('M4V','*.m4v'),
            ('FLV','*.flv')
        )
        self.erlaubte_dateiendungen = (
          'mp4' , 'MP4',  'WEBM', 'webm', 'OGG', 'ogg', 'OGV', 'ogv', 
          'MPG',  'mpg',  'MPEG', 'mpeg', 'AVI', 'avi', 'WMV', 'wmv', 
          'QT',   'qt',   'MOV',  'mov',  'MKV', 'mkv', 'M4V', 'm4v', 
          'FLV',  'flv')
        ## Container-Inhaltsverzeichnis moov an den Dateianfang setzen
        self.movflags=StringVar()
        self.movflags='faststart', 
        ## Video-Codec h264
        self.vcodec=StringVar()
        self.vcodec='libx264',
        ## Audiocodec AAC (wie mp3, wie lame) 
        self.acodec=StringVar()
        self.acodec='aac',
        ## auch mit nicht-iOS-Geräten kompatibles Farbprofil
        self.pix_fmt=StringVar()
        self.pix_fmt='yuv420p'
        ## Ziel-Containerformat
        self.dateiendung_neu='mp4'
        ## Styles
        self.prim_bg='#0d6efd' ## '#00a8d5'
        self.sec_bg='#dc3545' ## '#00a8d5'  #b02a37
        self.sec_hover='#b02a37' ## '#00a8d5'  #b02a37
        self.prim_fg='#ffffff' ## '#ffffff'
        self.main_bg='#ffffff' ## '#ffffff'
        self.prim_hoverbackground='#0a58ca'
        self.prim_hoverforeground=self.main_bg
        self.fenster.configure(background=self.main_bg)
        self.fenster.option_add('*Dialog.msg.font', 'Arial 12')
        self.fenster.option_add('*Dialog.msg.background', self.main_bg)
        self.fenster.option_add('*Dialog.msg.activeBackground', self.main_bg)
        self.fenster.option_add('*Button.relief', 'flat')
        self.fenster.option_add('*Button.foreground', self.prim_fg)
        self.fenster.option_add('*Button.disabledforeground', '#666666')
        self.fenster.option_add('*Button.font', 'Arial 12')
        self.fenster.option_add('*Radiobutton.background', self.main_bg)
        self.fenster.option_add('*Radiobutton.foreground', '#111111')
        self.fenster.option_add('*Radiobutton.font', 'Arial 12')
        self.fenster.option_add('*Label.background', self.main_bg)
        self.fenster.option_add('*Label.foreground', '#111111')
        self.fenster.option_add('*Label.disabledforeground', '#666666')
        self.fenster.option_add('*Label.font', 'Arial 12')
        self.fenster.option_add('*Entry.background', self.main_bg)
        self.fenster.option_add('*Entry.foreground', '#111111')
        self.fenster.option_add('*Entry.selectedbackground', self.main_bg)
        self.fenster.option_add('*Entry.selectedforeground', '#111111')
        self.fenster.option_add('*Entry.relief', 'flat')
        self.fenster.option_add('*Entry.font', 'Arial 12')
        ## GUI-Elemente definieren und erzeugen
        self.l_programmname = Label(master=self.fenster, 
                                    text='Video umwandeln in drei Schritten:')        
        self.b_dateiauswahl = Button(master=self.fenster, 
                                     text='1. Video-Datei  📹',
                                     command=self.datei_auswaehlen, 
                                     background=self.prim_bg) ##  📹 🎬 🎦
        self.e_dateiname = Entry(master=self.fenster, 
                                 readonlybackground='#ffffff', 
                                 highlightthickness=0)
        self.e_dateiname.insert(0, 'noch keine ausgewählt')
        self.e_dateiname.config(state='readonly')
        self.l_zielaufloesung = Label(master=self.fenster, 
                                      text='2. Zielauflösung', 
                                      state='disabled')
        self.cb_1080p = Radiobutton(master=self.fenster, 
                                    text='1920x1080', 
                                    variable=self.aufloesung, 
                                    value='1080p', 
                                    state='disabled', 
                                    highlightthickness=0)
        self.cb_720p = Radiobutton(master=self.fenster, 
                                   text='1280x720', 
                                   variable=self.aufloesung, 
                                   value='720p', 
                                   state='disabled', 
                                   highlightthickness=0)
        self.cb_480p = Radiobutton(master=self.fenster, 
                                   text='854x480', 
                                   variable=self.aufloesung, 
                                   value='480p', 
                                   state='disabled', 
                                   highlightthickness=0)
        self.cb_360p = Radiobutton(master=self.fenster, 
                                   text='640x360', 
                                   variable=self.aufloesung, 
                                   value='360p', 
                                   state='disabled', 
                                   highlightthickness=0)   
        self.cb_240p = Radiobutton(master=self.fenster, 
                                   text='426x240', 
                                   variable=self.aufloesung, 
                                   value='240p', 
                                   state='disabled', 
                                   highlightthickness=0)     
        self.b_verzeichnisauswahl = Button(master=self.fenster, 
                                    text='3. Ziel-Verzeichnis  📂', 
                                    command=self.verzeichnis_auswaehlen, 
                                    state='disabled')
        self.e_verzeichnis = Entry(master=self.fenster, 
                                   readonlybackground='#ffffff', 
                                   highlightthickness=0)
        self.e_verzeichnis.insert(0, 'noch keines ausgewählt')
        self.e_verzeichnis.config(state='readonly')
        self.b_umwandeln = Button(master=self.fenster, 
                                text='Umwandeln  ↷', 
                                command=self.umwandeln, 
                                state='disabled') ## 🏁 ✔ ☑  ↷
        self.b_schliessen = Button(master=self.fenster, 
                                   text='Schließen  ×', 
                                   command=self.fenster.destroy, 
                                   background=self.sec_bg) ## × ✖ ❌
        ## GUI-Elemente im Fenster anzeigen und Hauptschleife aufrufen
        self.layout()
        self.fenster.mainloop()
                
    def layout(self):
        ## Anordnung
        self.l_programmname.pack(fill='x', expand='yes')
        self.b_dateiauswahl.pack(fill='x', expand='yes', padx=5, pady=5)
        self.e_dateiname.pack(fill='x', expand='yes', padx=5, pady=5)
        self.l_zielaufloesung.pack(fill='x', expand='yes', padx=5, pady=5)
        self.cb_1080p.pack(anchor='w') 
        self.cb_720p.pack(anchor='w')
        self.cb_480p.pack(anchor='w')
        self.cb_360p.pack(anchor='w')
        self.cb_240p.pack(anchor='w')
        self.b_verzeichnisauswahl.pack(fill='x', expand='yes', padx=5, pady=5)
        self.e_verzeichnis.pack(fill='x', expand='yes', padx=5, pady=5)
        self.b_umwandeln.pack(fill='x', expand='yes', padx=5, pady=5)
        self.b_schliessen.pack(fill='x', expand='yes', pady=5, padx=5)
        ## Hover-Effekt bei Buttons
        self.b_dateiauswahl.bind('<Enter>', self.on_enter)
        self.b_dateiauswahl.bind('<Leave>', self.on_leave)
        self.b_verzeichnisauswahl.bind('<Enter>', self.on_enter)
        self.b_verzeichnisauswahl.bind('<Leave>', self.on_leave)
        self.b_umwandeln.bind('<Enter>', self.on_enter)
        self.b_umwandeln.bind('<Leave>', self.on_leave)
        self.b_schliessen.bind('<Enter>', self.on_enter_danger)
        self.b_schliessen.bind('<Leave>', self.on_leave)
        
    def on_enter(self, e):
        #print('on_enter')
        e.widget['activebackground'] = self.prim_hoverbackground
        e.widget['activeforeground'] = self.prim_hoverforeground
        
    def on_enter_danger(self, e):
        #print('on_enter')
        e.widget['activebackground'] = self.sec_hover
        e.widget['activeforeground'] = self.prim_hoverforeground

    def on_leave(self, e):
        #print('on_leave')
        e.widget['activebackground'] = self.prim_bg
        e.widget['activeforeground'] = self.prim_hoverforeground
        
    def datei_auswaehlen(self):
        #print('datei_auswaehlen')
        self.quelldatei = StringVar()
        self.quelldatei = askopenfilename(initialdir = '.',
                                          title = 'Videodatei auswählen',
                                          filetypes = self.quelldateitypen)
        if (str(self.quelldatei) == '()'):
            #print('Es wurde keine Datei ausgewählt')
            self.hinweis_keineauswahl = messagebox.showinfo('Hinweis', 'Es wurde keine Datei ausgewählt!')
        else:
            #print('ok - weiter mit: '+str(self.quelldatei))
            self.pfad_betriebssystem = os.path.normpath(self.quelldatei)
            #print(self.pfad_betriebssystem)
            self.dateiendung = os.path.basename(self.pfad_betriebssystem).rsplit('.', 1)[1]
            #print('Dateiendung: '+str(self.dateiendung))
            if not str(self.dateiendung) in self.erlaubte_dateiendungen:
                #print('Die Endung '+str(self.dateiendung)+' ist nicht erlaubt!')
                self.hinweis_ungeltigerdateityp = messagebox.showinfo('Hinweis', 'Die gewählte Datei besitzt kein gültiges Eingabeformat!')
            else:
                self.e_dateiname.config(state='normal')
                self.e_dateiname.delete(0, 'end')
                self.e_dateiname.insert(0, os.path.basename(self.pfad_betriebssystem))
                self.e_dateiname.config(state='readonly')
                self.l_zielaufloesung.config(state='normal')
                self.cb_1080p.config(state='normal')
                self.cb_720p.config(state='normal')
                self.cb_480p.config(state='normal')
                self.cb_360p.config(state='normal')
                self.cb_240p.config(state='normal')
                self.b_verzeichnisauswahl.config(state='normal', background=self.prim_bg, foreground='#ffffff')
    
    def verzeichnis_auswaehlen(self):
        #print('verzeichnis_auswaehlen')
        self.zielverzeichnis = StringVar()
        self.zielverzeichnis = askdirectory(initialdir = sys.path[0],
                                            title = 'Zielverzeichnis auswählen')
        if (str(self.zielverzeichnis) == '()' or str(self.zielverzeichnis) == '.'):
            #print('Es wurde kein Verzeichnis ausgewählt')
            self.hinweis_keinverzeichnis = messagebox.showinfo('Hinweis', 'Es wurde kein Verzeichnis ausgewählt!')
        else:
            #print('ok - Zielverzeichnis: '+str(self.zielverzeichnis))
            self.verzeichnispfad_betriebssystem = os.path.normpath(self.zielverzeichnis)
            #print(self.verzeichnispfad_betriebssystem)
            self.e_verzeichnis.config(state='normal')
            self.e_verzeichnis.delete(0, 'end')
            self.e_verzeichnis.insert(0, str(self.verzeichnispfad_betriebssystem))
            self.e_verzeichnis.config(state='readonly')
            self.b_umwandeln.config(state='normal', background=self.prim_bg, foreground='#ffffff')
        
    
    def umwandeln(self):
        #print('umwandeln')
        self.auswahl = self.aufloesung.get()
        print(self.auswahl)
        self.deaktivieren()
        
        self.hinweis_umwandeln_gestartet = messagebox.askokcancel('Hinweis', 'Die Umwandlung der Quelldatei '+str(os.path.basename(self.pfad_betriebssystem))+' mit der Zielauflösung '+str(self.auswahl)+' wird mit Klicken auf "OK" gestartet.\n\n⚠ Die Umwandlung kann nicht abgebrochen werden! Auch wenn das Programmfenster geschlossen wird, läuft die Umwandlung im Hintergrund weiter!')
        
        #print(self.hinweis_umwandeln_gestartet)
        
        if (self.hinweis_umwandeln_gestartet == True):
            self.grundpfad=self.verzeichnispfad_betriebssystem
            self.dateiname_netto=str(os.path.basename(self.pfad_betriebssystem).rsplit('.', 1)[0])+'_'+str(self.aufloesungen[self.auswahl][0])+'x'+str(self.aufloesungen[self.auswahl][1])
            
            self.dateiname_komplett=self.dateiname_netto+'.'+self.dateiendung_neu
            self.ausgabe=os.path.join(self.grundpfad, self.dateiname_komplett)
            #print(self.ausgabe)
            # Prüfen, ob eine gleichnamige Datei wie die Ausgabedatei bereits vorliegt und ob diese ggfs. überschrieben werden soll            
            if os.path.exists(self.ausgabe):
                #print('Achtung, diese Ausgabedatei besteht bereits, soll diese überschrieben werden?')
                # Rückfrage und ggfs. Abbrechen oder Umwandlung durchführen
                self.rueckfrage_ueberschreiben = messagebox.askokcancel('Hinweis', 'Die Ausgabedatei '+str(self.ausgabe)+' besteht bereits. Soll diese überschrieben werden?')
                if (self.rueckfrage_ueberschreiben):
                    self.umwandlung_durchfuehren()
            else:
                self.umwandlung_durchfuehren()
            
            self.aktivieren()
        else:
            self.aktivieren()
        
    def umwandlung_durchfuehren(self):
        #print('umwandlung_durchfueren')
        
        self.process1 = (
            ffmpeg
                .input(self.pfad_betriebssystem)
                .filter('scale', 
                        width=self.aufloesungen[self.auswahl][0], 
                        height=self.aufloesungen[self.auswahl][1])
                .output(self.ausgabe, 
                        movflags=self.movflags[0], 
                        vcodec=self.vcodec[0], 
                        acodec=self.acodec[0], 
                        pix_fmt='yuv420p')
                .overwrite_output()
                .run_async(pipe_stdout=True)
                #.run()
        )
        
    def deaktivieren(self):
        self.b_dateiauswahl.config(state='disabled')
        self.l_zielaufloesung.config(state='disabled')
        self.cb_1080p.config(state='disabled')
        self.cb_720p.config(state='disabled')
        self.cb_480p.config(state='disabled')
        self.cb_360p.config(state='disabled')
        self.cb_240p.config(state='disabled')
        self.b_verzeichnisauswahl.config(state='disabled')
        self.e_verzeichnis.config(state='disabled')
        self.b_umwandeln.config(state='disabled')
        
    def aktivieren(self):
        self.b_dateiauswahl.config(state='normal')
        self.l_zielaufloesung.config(state='normal')
        self.cb_1080p.config(state='normal')
        self.cb_720p.config(state='normal')
        self.cb_480p.config(state='normal')
        self.cb_360p.config(state='normal')
        self.cb_240p.config(state='normal')
        self.b_verzeichnisauswahl.config(state='normal')
        self.e_verzeichnis.config(state='normal')
        self.b_umwandeln.config(state='normal')



## Programm starten
v = Videoumwandler()
